﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Linq;
using System.IO;

using StockSharp.Algo.Candles;
using StockSharp.BusinessEntities;
using StockSharp.Messages;
using StockSharp.Algo.Storages;

using Newtonsoft.Json;

namespace StorageBugExample
{
    public partial class Form1 : Form
    {

        // this is the Poloniex candles read from JSON file
        List<PoloniexCandle> PolCandles { get; set; }  = new List<PoloniexCandle>();

        List<Candle> Candles { get; set; }  = new List<Candle>();
        List<Candle> LoadedCandles { get; set; } = new List<Candle>();
        Security Security { get; set; }
        StorageRegistry Storage { get; set; }

        string DataPath { get; } = @"c:\StockSharpData";

        public Form1()
        {
            InitializeComponent();

            var board = ExchangeBoard.Associated;

            // Create a new security for which we would like to store in S# format
            this.Security = new Security
            {
                Id = "BTCEUR@Poloniex",
                PriceStep = 0.00000001m,
                Decimals = 8,
                Board = board,
            };

            this.Storage = new StorageRegistry();
            ((LocalMarketDataDrive)Storage.DefaultDrive).Path = this.DataPath;
        }

      
        private void button2_Click(object sender, EventArgs e)
        {
            var timeFrame = new TimeSpan(0, 5, 0);
            var candleStorage = this.Storage.GetCandleStorage(typeof(TimeFrameCandle), this.Security,timeFrame, format: StorageFormats.Binary);

            candleStorage.Save(this.Candles);

            listBox1.Items.Add($"Saved {this.Candles.Count} CANDLE data saved with time frame={timeFrame} and format={StorageFormats.Binary}.");
        }

        private void button3_Click(object sender, EventArgs e)
        {
            var timeFrame = new TimeSpan(0, 5, 0);
            var candleStorage = this.Storage.GetCandleStorage(typeof(TimeFrameCandle), this.Security, timeFrame, format: StorageFormats.Binary);

            // loading ticks
            DateTime startTime = DateTime.Parse("2017-01-01 00:00:00");
            this.LoadedCandles = candleStorage.Load(startTime, DateTime.Now).ToList();

            listBox1.Items.Add($"Loaded {this.LoadedCandles.Count} candle data from {startTime} with time frame={timeFrame} and format={StorageFormats.Binary}.");

        }

        private void button4_Click(object sender, EventArgs e)
        {
            listBox1.Items.Add($"Will Compare {LoadedCandles.Count} Candles...");
            for (int i = 0; i < LoadedCandles.Count; i++)
            {
                if (this.Candles[i].OpenPrice != this.LoadedCandles[i].OpenPrice || this.Candles[i].ClosePrice != this.LoadedCandles[i].ClosePrice || this.Candles[i].TotalVolume != this.LoadedCandles[i].TotalVolume)
                {
                    listBox1.Items.Add($"Corrupt at i={i} Open={this.Candles[i].OpenPrice}:Loaded.Open={this.LoadedCandles[i].OpenPrice}, Close={this.Candles[i].ClosePrice}:Loaded.Close{this.LoadedCandles[i].ClosePrice}, Vol={this.Candles[i].TotalVolume}:Loaded.Vol={this.LoadedCandles[i].TotalVolume}");
                }
            }
            listBox1.Items.Add($"Compare Complete");
        }

        private void button5_Click(object sender, EventArgs e)
        {
            var timeFrame = new TimeSpan(0, 5, 0);
            var candleStorage = this.Storage.GetCandleStorage(typeof(TimeFrameCandle), this.Security, timeFrame, format: StorageFormats.Binary);

            // Delete existing storage
            candleStorage.Delete();

            listBox1.Items.Add($"Deleted all Binary CANDLE data from {this.DataPath}.");
        }

        
        // Read JSON and convert
        private void button6_Click(object sender, EventArgs e)
        {
            using (StreamReader r = new StreamReader("PoloniexETH_USD.json"))
            {
                string json = r.ReadToEnd();
                this.PolCandles = JsonConvert.DeserializeObject<List<PoloniexCandle>>(json);
            }

            listBox1.Items.Add($"Read JSON File.. Found {this.PolCandles.Count} candles..");

            // Start a new Candle list
            this.Candles = new List<Candle>();

            List<string> lines = new List<string>();
            int i = 0;
            foreach (PoloniexCandle polCandle in this.PolCandles)
            {
                var scandle = new TimeFrameCandle();
                scandle.TimeFrame = TimeSpan.FromSeconds(300);
                scandle.OpenTime = DateTimeOffset.FromUnixTimeSeconds(polCandle.date);
                scandle.CloseTime = scandle.OpenTime.AddSeconds(300);
                //scandle.HighTime = DateTimeOffset.FromUnixTimeSeconds(polCandle.date);
                //scandle.LowTime = DateTimeOffset.FromUnixTimeSeconds(polCandle.date);
                scandle.State = CandleStates.Finished;
                scandle.OpenPrice = polCandle.open;
                if (scandle.OpenPrice > 500m)
                {
                    listBox1.Items.Add($"Candle at datetime={scandle.OpenTime} has open price={scandle.OpenPrice.ToString("0.00")}");
                }
                scandle.ClosePrice = polCandle.close;
                scandle.TotalVolume = polCandle.volume;
                scandle.HighPrice = polCandle.high;
                scandle.LowPrice = polCandle.low;
                scandle.Security = this.Security;
                this.Candles.Add(scandle);

                lines.Add($"i={i}: OpenTime={scandle.OpenTime}, OpenPrice={scandle.OpenPrice} ClosePrice={scandle.ClosePrice} High={scandle.HighPrice} Low={scandle.LowPrice} Vol={scandle.TotalVolume}");
                i++;
            }
            
            // WriteAllLines creates a file, writes a collection of strings to the file,
            // and then closes the file.  You do NOT need to call Flush() or Close().
            System.IO.File.WriteAllLines("debug.txt", lines);

            listBox1.Items.Add($"Converted {this.Candles.Count} candles with time frame={TimeSpan.FromSeconds(300)} to s# Candles...");
        }
    }
}
