﻿namespace StockSharp.AlfaDirect
{
	using System;
	using System.Collections.Generic;
	using System.Threading;

	using Ecng.Common;
	using Ecng.ComponentModel;

	using StockSharp.BusinessEntities;
	using StockSharp.Algo;
	using StockSharp.Algo.Candles;
	using StockSharp.Algo.Logging;

	/// <summary>
	/// Реализация интерфейса <see cref="ITrader"/>, предоставляющая шлюз взаимодействия с системой AlfaDirect.
	/// </summary>
	public class AlfaTrader : BaseTrader
	{
		//private const string _logTag = "AlfaTrader";

	    private AlfaWrapper _wrapper;
	    private readonly object _wrapperCreationMutex = new object();

		/// <summary>
		/// Отложенная инициализация AlfaWrapper
		/// </summary>
		private AlfaWrapper Wrapper
        {
            get
            {
                if (_wrapper == null)
                {
                    lock (_wrapperCreationMutex)
                    {
                        if (_wrapper == null)
                        {
                            _wrapper = new AlfaWrapper();
                            _wrapper.ProcessSecurities += OnProcessSecurities;
                            _wrapper.ProcessOrders += OnProcessOrders;
                            _wrapper.ProcessPortfolios += OnProcessPortfolios;
                            _wrapper.ProcessQuotes += OnProcessQuotes;
                            _wrapper.ProcessTrades += OnProcessTrades;
                        	_wrapper.ProcessPositions += OnProcessPositions;
							_wrapper.ProcessMyTrades += OnProcessMyTrades;

							this.AddInfoLog("AlfaDirect v. {0}", _wrapper.Version);
                        }
                    }
                }

                return _wrapper;
            }
        }

		private void OnProcessTrades(string tableParams, string data)
		{
			// params: paper_no = 36955
			var paperNo = tableParams.Split(new[] { '=' })[1].Trim();

			var security = GetSecurity(paperNo);

			this.AddInfoLog("OnProcessTrades(): {0}", security.Id);

			// "trd_no, paper_no, qty, price, ts_time, b_s";
			// 274847059|36955|5|9930|10:10:11|0|
			// 0 - покупка, 1 - продажа

			//Logger.Debug(data);

			var trades = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				foreach (var tradeStr in trades)
				{
					var details = tradeStr.Split('|');

					GetTrade(details[0].To<long>(), id =>
					{
						var trade = EntityFactory.CreateTrade(id);

						trade.Security = security;
						trade.OrderDirection = AlfaUtils.OrderDirectionsFromAlfa(details[5].To<int>());
						trade.Price = details[3].To<decimal>();
						trade.Time = details[4].To<DateTime>();
						trade.Volume = details[2].To<int>();

						return trade;
					});
				}
			});
		}

		private void OnProcessQuotes(string tableParams, string data)
		{
			// tableParams: paper_no = 36955
			var paperNo = tableParams.Split(new[] { '=' })[1].Trim();
			var security =  GetSecurity(paperNo);

			this.AddInfoLog("OnProcessQuotes(): {0}", security.Id);

			var quotes = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				var bids = new List<Quote>();
				var asks = new List<Quote>();

				foreach (var quoteStr in quotes)
				{
					// quoteStr: 0|10138|14|

					var details = quoteStr.Split('|');
					var sellQty = details[0].To<int>();
					var price = details[1].To<decimal>();
					var buyQty = details[2].To<int>();

					var quote = new Quote
					{
						Price = price,
						Security = security
					};

					if (sellQty == 0)
					{
						quote.OrderDirection = OrderDirections.Buy;
						quote.Volume = buyQty;

						bids.Insert(0, quote);
					}
					else if (buyQty == 0)
					{
						quote.OrderDirection = OrderDirections.Sell;
						quote.Volume = sellQty;

						asks.Add(quote);
					}

					// If the sellQty and buyQty are 0 - that is our limit order 
					// which is not a part of the market depth. Just skip it.
				}

				var marketDepth = GetMarketDepth(security);
				marketDepth.Update(bids, asks, true);

				RaiseQuotesChanged(new[] { marketDepth });

				security.BestAsk = marketDepth.BestAsk;
				security.BestBid = marketDepth.BestBid;
				RaiseSecuritiesChanged(new[] { security });
			});
		}

		private void OnProcessPortfolios(string data)
		{
			this.AddInfoLog("OnProcessPortfolios() {0}", data);

			var portfolios = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				foreach (var str in portfolios)
				{
					var details = str.Split('|');

					GetPortfolio(details[0],
					name =>
					{
						var portfolio = EntityFactory.CreatePortfolio(name);
						portfolio.CurrentAmount = details[1].To<decimal>();
						portfolio.BeginAmount = details[2].To<decimal>();
						return portfolio;
					},
					portfolio =>
					{
						portfolio.CurrentAmount = details[1].To<decimal>();
						portfolio.BeginAmount = details[2].To<decimal>();
					});
				}
			});
		}

		private void OnProcessOrders(string data)
		{
			this.AddInfoLog("OnProcessOrders() {0}", data);

			var ordersStr = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				foreach (var str in ordersStr)
				{
					var details = str.Split('|');

					GetOrder(details[0].To<long>(), id =>
					{
						this.AddInfoLog("create order: id = {0}", id);

						var order = EntityFactory.CreateOrder(id);

						// TODO: Use portfolio instead of the account.
						order.Portfolio = GetPortfolio(details[1].Split("-")[0]);
						order.Security = GetSecurity(details[2]);

						// TODO: set order type (Limit/Market/Conditional)
						// TODO: RaiseNewOrder)

						return order;
					},
					order =>
					{
						this.AddInfoLog("update order: id = {0}", order.Id);

						order.Direction = AlfaUtils.OrderDirectionsFromAlfa(details[4]);
						order.Price = details[5].To<decimal>();
						order.Volume = details[6].To<Int32>();
						order.Balance = details[7].To<Int32>();
						order.Time = DateTime.Parse(details[8]);

						switch (details[3]) // статус
						{
							case "O": // активная
							{
								order.State = OrderStates.Active;
								break;
							}
							case "M": // исполнена
							{
								order.State = (order.Balance == 0) ? OrderStates.Done : OrderStates.Active;
								break;
							}
							case "W": // удалена
							{
								order.State = OrderStates.Done;
								order.CancelTime = Wrapper.MarketTime;
								break;
							}
							default:
							{
								this.AddInfoLog("Order status {0} is not taken into account", details[3]);
								break;
							}
						}

						if (details[9].Length > 0)
						{
							// Read transaction id from the comments
                            // TODO: Fix how to update orders from the terminal and preserve same tr. id.
							//order.TransactionId = details[9].To<long>();
						}

						// TODO: RaiseOrdersChanged
					});
				}
			});
		}

		/// <summary>
		/// Creation/Update of the securities.
		/// </summary>
		/// <param name="data">AD raw data</param>
		/// <param name="create">Indicates if it's creation or update of the securitites</param>
		private void OnProcessSecurities(string data, bool create = false)
		{
			this.AddInfoLog("OnProcessSecurities {0}", data);

			var secutities = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				foreach (var s in secutities)
				{
					var securityInfo = s;

					var details = securityInfo.Split('|');

					if(details[1].IsEmpty())
					{
						continue;
					}

					if (create)
					{
						GetSecurity(details[0], name =>
						{
							this.AddInfoLog("Security create : {0}", securityInfo);

							var security = EntityFactory.CreateSecurity(details[4] + "@" + details[5]);
							security.ExtensionInfo = new Dictionary<object, object>();

							security.Name = details[1];
							security.ShortName = details[1];
							security.ExpiryDate = DateTime.Parse(details[2]);
							security.Code = details[4];
							security.State = AlfaUtils.SecurityStateFromAlfa(details[3]);
							var exCode = Wrapper.GetExchangeCode(details[5]);
							security.Exchange = AlfaUtils.ExchangeCodeToExchange(exCode);
							security.MarginBuy = details[7].To<decimal>();
							security.MarginSell = details[8].To<decimal>();

							security.SetPaperNo(details[0]);
							security.SetCurrency(details[6]);
							security.SetPlaceCode(details[5]);

							decimal priceStep, priceStepCost;
							Wrapper.GetPriceStepInfo(security, out priceStep, out priceStepCost);
							security.MinStepSize = priceStep;

							if (priceStepCost == 0)
							{
								this.AddWarningLog("Стоимость шага цены равна нулю для {0}.", security.Name);
							}
							else
							{
								security.MinStepPrice = priceStepCost;	
							}

							security.Type = AlfaUtils.BoardCodeToSecurityType(details[9]);
                            security.ExtensionInfo["SecurityOpenedPositionsPrice"] = details[10].To<decimal>();
							return security;
						}, null);
					}
					else
					{
						GetSecurity(details[0], security =>
						{
							this.AddInfoLog("Security update {0}.", security.Id);

							security.BestAsk = new Quote
							{
								Price = details[1].To<decimal>(),
								Volume = details[2].To<int>(),
								Security = security,
								OrderDirection = OrderDirections.Sell
							};

							security.BestBid = new Quote
							{
								Price = details[3].To<decimal>(),
								Volume = details[4].To<int>(),
								Security = security,
								OrderDirection = OrderDirections.Buy
							};

							security.LowPrice = details[5].To<decimal>();
							security.HighPrice = details[6].To<decimal>();
                            security.ExtensionInfo["SecurityOpenedPositionsPrice"] = details[7].To<decimal>();

							// TODO: Update security state
							// TODO: Add missing fields
						});
					}
				}
			});

			// TODO: Add missing Security fields
			/*
			SecurityFields = "paper_no, ts_p_code, board_code, sell, sell_qty, buy, buy_qty, min_deal, max_deal, lot_size, decimals, status, price_step, p_code, volatility, theor_price, mat_date";
			
			security.MinLotSize = details[9].To<int>();
			security.MinStepSize = details[10].To<decimal>();
			//security.State = details[11].To<int>();
			security.MinStepPrice = details[12].To<decimal>();
			security.UnderlyingSecurityId = details[13];
			security.Volatility = details[14].To<decimal>();
			security.TheorPrice = details[15].To<decimal>();
			*/
		}

		/// <summary>
		/// Update positions
		/// </summary>
		private void OnProcessPositions(string data)
		{
			this.AddInfoLog("OnProcessPositions() {0}", data);

			var positionsStr = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				foreach (var str in positionsStr)
				{
					var details = str.Split('|');

					if(details[3] == "money")
					{
						continue;
					}

					GetPosition(GetPortfolio(details[0]), GetSecurity(details[1]),
					(portfolio, security) =>
					{
						var position = EntityFactory.CreatePosition(portfolio, security);
						position.CurrentValue = details[2].To<long>();
						return position;
					},
					position =>
					{
						position.CurrentValue = details[2].To<long>();
					});
				}
			});
		}

		/// <summary>
		/// Update my trades
		/// </summary>
		/// <param name="data"></param>
		private void OnProcessMyTrades(string data)
		{
			this.AddInfoLog("OnProcessMyTrades() {0}", data);

			var mytradesStr = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

			ProcessEvents(() =>
			{
				foreach (var str in mytradesStr)
				{
					var details = str.Split('|');

					AddMyTrade(details[1].To<long>(), 0, details[0].To<long>(), id =>
					{
					   	var trade = EntityFactory.CreateTrade(id);

						trade.Price = details[4].To<decimal>();
						trade.Security = GetSecurity(details[3]);
						trade.Time = DateTime.Parse(details[7]);
						trade.Volume = details[5].To<int>();
						trade.OrderDirection = AlfaUtils.OrderDirectionsFromAlfa(details[6]);

						return trade;
					}, t => { });
				}
			});
		}

		/// <summary>
		/// Зарегистрировать заявку на бирже.
		/// </summary>
		/// <param name="order">Заявка, содержащая информацию для регистрации.</param>
		protected override void OnRegisterOrder(Order order)
		{
			// есть в базовом классе
			//Logger.Debug("OnRegisterOrder: tr.id = " + order.TransactionId, _logTag);

			// OnProcessOrders will read transaction id from the comment
			order.Comment = order.TransactionId.ToString();

            Wrapper.RegisterOrder(order);
		}

		/// <summary>
		/// Перерегистрировать заявку на бирже.
		/// </summary>
		/// <param name="oldOrder">Заявка, которую нужно снять.</param>
		/// <param name="newOrder">Новая заявка, которую нужно зарегистрировать.</param>
		protected override void OnReRegisterOrder(Order oldOrder, Order newOrder)
		{
			if (oldOrder.Security.Exchange == Exchange.Rts)
			{
                Wrapper.ReRegisterOrder(oldOrder, newOrder);
			}
			else
			{
				base.OnReRegisterOrder(oldOrder, newOrder);
			}
		}

		/// <summary>
		/// Отменить заявку на бирже.
		/// </summary>
		/// <param name="order">Заявка, которую нужно отменять.</param>
		protected override void OnCancelOrder(Order order)
		{
			// есть в базовом классе
			//Logger.Debug("OnCancelOrder(): id " + order.Id + ", tr.id: " + order.TransactionId, _logTag);

			if (order.Id == 0)
			{
				this.AddErrorLog("Failed to cancel an order which is not active yet.");
				return;
			}

            Wrapper.CancelOrder(order);
		}

		/// <summary>
		/// Отменить группу заявок на бирже по фильтру.
		/// </summary>
		/// <param name="isStopOrder">True, если нужно отменить только стоп-заявки, false - если только обычный и null - если оба типа.</param>
		/// <param name="portfolio">Портфель. Если значение равно null, то портфель не попадает в фильтр снятия заявок.</param>
		/// <param name="direction">Направление заявки. Если значение равно null, то направление не попадает в фильтр снятия заявок.</param>
		/// <param name="classCode">Код класса. Если переданная строка пустая, то код не попадает в фильтр снятия заявок.</param>
		/// <param name="security">Инструмент. Если значение равно null, то инструмент не попадает в фильтр снятия заявок.</param>
		public override void CancelOrders(bool? isStopOrder = null, Portfolio portfolio = null, OrderDirections? direction = null, string classCode = null, Security security = null)
		{
			Wrapper.CancelOrders(isStopOrder, portfolio, direction, classCode, security);
		}

		/// <summary>
		/// Подключиться к торговой системе.
		/// </summary>
		protected override void OnConnect()
		{
			this.AddInfoLog("OnConnect()");

            if (Wrapper.IsConnected)
			{
				RaiseConnected();
			}
		}

		/// <summary>
		/// Отключиться от торговой системы.
		/// </summary>
		protected override void OnDisconnect()
		{
			this.AddInfoLog("OnDisconnect()");
		}

		/// <summary>
		/// Начать получать новую информацию по портфелю.
		/// </summary>
		/// <param name="portfolio">Портфель, по которому необходимо начать получать новую информацию.</param>
		public override void RegisterPortfolio(Portfolio portfolio)
		{
			this.AddInfoLog("RegisterPortfolio: {0}", portfolio.Name);

            Wrapper.RegisterPortfolio(portfolio);
		}

		/// <summary>
		/// Остановить получение новой информации по портфелю.
		/// </summary>
		/// <param name="portfolio">Портфель, по которому необходимо остановить получение новой информации.</param>
		public override void UnRegisterPortfolio(Portfolio portfolio)
		{
			this.AddInfoLog("UnRegisterPortfolio: {0}", portfolio.Name);

			Wrapper.UnRegisterPortfolio(portfolio);
		}

		/// <summary>
		/// Начать получать новую информацию (например, <see cref="P:StockSharp.BusinessEntities.Security.LastTrade"/> или <see cref="P:StockSharp.BusinessEntities.Security.BestBid"/>) по инструменту.
		/// </summary>
		/// <param name="security">Инструмент, по которому необходимо начать получать новую информацию.</param>
		public override void RegisterSecurity(Security security)
		{
			this.AddInfoLog("RegisterSecurity: {0}", security.Id);

            Wrapper.RegisterSecurity(security);
		}

		/// <summary>
		/// Остановить получение новой информации.
		/// </summary>
		/// <param name="security">Инструмент, по которому необходимо остановить получение новой информации.</param>
		public override void UnRegisterSecurity(Security security)
		{
			this.AddInfoLog("UnRegisterSecurity: {0}", security.Id);

			Wrapper.UnRegisterSecurity(security);
		}

		/// <summary>
		/// Начать получать котировки (стакан) по инструменту.
		/// Значение котировок можно получить через метод <see cref="M:StockSharp.BusinessEntities.ITrader.GetMarketDepth(StockSharp.BusinessEntities.Security)"/>.
		/// </summary>
		/// <param name="security">Инструмент, по которому необходимо начать получать котировки.</param>
		public override void RegisterQuotes(Security security)
		{
			this.AddInfoLog("RegisterQuotes: {0}", security.Id);

            Wrapper.RegisterQuotes(security);
		}

		/// <summary>
		/// Остановить получение котировок по инструменту.
		/// </summary>
		/// <param name="security">Инструмент, по которому необходимо остановить получение котировок.</param>
		public override void UnRegisterQuotes(Security security)
		{
			this.AddInfoLog("UnRegisterQuotes: {0}", security.Id);

			Wrapper.UnRegisterQuotes(security);
		}

		/// <summary>
		/// Начать получать сделки (тиковые данные) по инструменту.
		/// Новые сделки будут приходить через событие <see cref="E:StockSharp.BusinessEntities.ITrader.NewTrades"/>.
		/// </summary>
		/// <param name="security">Инструмент, по которому необходимо начать получать сделки.</param>
		public override void RegisterTrades(Security security)
		{
			this.AddInfoLog("RegisterTrades: {0}", security.Id);

            Wrapper.RegisterTrades(security);
		}

		/// <summary>
		/// Остановить получение сделок (тиковые данные) по инструменту.
		/// </summary>
		/// <param name="security">Инструмент, по которому необходимо остановить получение сделок.</param>
		public override void UnRegisterTrades(Security security)
		{
			this.AddInfoLog("UnRegisterTrades: {0}", security.Id);

			Wrapper.UnRegisterTrades(security);
		}

		/// <summary>
		/// Запустить экспорт данных из торговой системы в программу (получение портфелей, инструментов, заявок и т.д.).
		/// </summary>
		public override void StartExport()
		{
			this.AddInfoLog("StartExport()");

			if (IsExportRunning)
			{
				this.AddWarningLog("Export is already running");
				return;
			}

			base.StartExport();

			Wrapper.ReadPortfolios();
			Wrapper.ReadSecurities();

			Wrapper.ReadOrders();
			Wrapper.ReadMyTrades();

            Wrapper.StartExportOrders();
			Wrapper.StartExportPositions();

			// TODO: start export my trades
		}

		/// <summary>
		/// Остановить экспорт данных из торговой системы в программу, запущенный через <see cref="M:StockSharp.BusinessEntities.ITrader.StartExport"/>.
		/// </summary>
		public override void StopExport()
		{
			this.AddInfoLog("StopExport()");

			if (!IsExportRunning)
			{
				this.AddWarningLog("Export is already stoped");
				return;
			}

			base.StopExport();

            Wrapper.StopExportOrders();
			Wrapper.StopExportPositions();
		}

		/// <summary>
		/// Получить исторические данные (тайм-фрейм свечки) от сервера Альфа-Директ.
		/// </summary>
		/// <param name="security">Инструмент, для которого необходимо получить исторические данные.</param>
		/// <param name="timeFrame">Тайм-фрейм.</param>
		/// <param name="range">Диапазон времени, для которого нужно получить данные.</param>
		/// <returns>Исторические данные.</returns>
		public IEnumerable<Candle> GetHistoryData(Security security, AlfaTimeFrames timeFrame, Range<DateTime> range)
		{
			if (security == null)
				throw new ArgumentNullException("security");

			if (timeFrame == null)
				throw new ArgumentNullException("timeFrame");

			if (range == null)
				throw new ArgumentNullException("range");

			this.AddInfoLog("GetHistoryData: security = {0}", security.Id);

			var data = _wrapper.GetHistoryData(security, timeFrame, range);
			this.AddInfoLog(data);

			var result = new List<Candle>();

			var candlesStr = data.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

// ReSharper disable LoopCanBeConvertedToQuery
			foreach (var str in candlesStr)
// ReSharper restore LoopCanBeConvertedToQuery
			{
				var details = str.Split('|');

				var candle = new TimeFrameCandle
				{
				    Security = security,
				    OpenPrice = details[1].To<decimal>(),
					HighPrice = details[2].To<decimal>(),
					LowPrice = details[3].To<decimal>(),
					ClosePrice = details[4].To<decimal>(),
				    TotalVolume = details[5].To<int>(),
				    TimeFrame = (TimeSpan) timeFrame,
					Time = DateTime.Parse(details[0])
				};

				result.Add(candle);
			}

			return result;
		}

		#region ITrader Properties

		/// <summary>
		/// Получить биржевое время. Значение зависит от <see cref="P:StockSharp.Algo.BaseTrader.MarketTimeOffset"/>.
		/// </summary>
		public override DateTime MarketTime
		{
			get { return Wrapper.MarketTime; }
		}

		/// <summary>
		/// Проверить соединение.
		/// </summary>
		public override bool IsConnected
		{
			get { return Wrapper.IsConnected; }
		}

		#endregion // ITrader Properties

		/// <summary>
		/// Автоматически авторизоваться в терминале Альфа-Директ.
		/// </summary>
		/// <param name="login">Логин.</param>
		/// <param name="password">Пароль.</param>
		public void Login(string login, string password)
		{
			if (login.IsEmpty())
			{
				throw new ArgumentNullException("login");
			}

			if (password.IsEmpty())
			{
				throw new ArgumentNullException("password");
			}

			if (IsConnected)
			{
				this.AddInfoLog("Терминал уже подключен.");
				return;
			}

			ThreadPool.QueueUserWorkItem(obj => new AlfaTerminal().Login(login, password, this));
		}
	}
}